(function () {
  'use strict';

  const PLUGIN_ID = kintone.$PLUGIN_ID;

  function isEnabled() {
    const config = kintone.plugin.app.getConfig(PLUGIN_ID);
    return config.enabled === 'true';
  }

  kintone.events.on('app.record.index.show', async function () {
    if (!isEnabled()) return;
    if (document.getElementById('simple-compact-search')) return;

    const header = kintone.app.getHeaderMenuSpaceElement();
    if (!header) return;

    /* =========================
     * UI
     * ========================= */
const area = document.createElement('div');
area.id = 'simple-compact-search';

// ★ 追加：中身用ラッパー
const inner = document.createElement('div');
inner.className = 'lookup-refetch-inner';

const select = document.createElement('select');
select.className = 'lookup-refetch-select';

const keyInput = document.createElement('input');
keyInput.type = 'text';
keyInput.placeholder = '参照キーを入力';
keyInput.className = 'w-name';

const button = document.createElement('button');
button.textContent = '一括で再取得';
button.className = 'lookup-refetch-button';

const progress = document.createElement('div');
progress.id = 'lookup-progress';

// ★ inner にまとめる
inner.append(select, keyInput, button, progress);
area.append(inner);
header.appendChild(area);

    /* =========================
     * ルックアップ取得
     * ========================= */
    const lookupFields = await fetchLookupFields();
    populateSelect(select, lookupFields);

    /* =========================
     * 実行
     * ========================= */
    button.onclick = async () => {
      const fieldCode = select.value;
      const keyValue = keyInput.value.trim();

      // 初期化
      progress.className = '';
      progress.textContent = '';

      /* ---- バリデーション ---- */
      if (!fieldCode) {
        progress.className = 'lookup-error';
        progress.textContent = 'ルックアップフィールドを選択してください';
        return;
      }

      if (!keyValue) {
        progress.className = 'lookup-error';
        progress.textContent = '参照キーを入力してください';
        keyInput.focus();
        return;
      }

      button.disabled = true;
      progress.textContent = '確認中…';

      let hasError = false;
      let updatedCount = 0;

      try {
        const records = await fetchRecordsByKey(fieldCode, keyValue);

        if (records.length === 0) {
          progress.className = 'lookup-error';
          progress.textContent = '該当するレコードはありません';
          return;
        }

        const ok = confirm(
          `参照キー「${keyValue}」を参照している\n` +
          `${records.length}件のレコードを再取得します。\n\n` +
          `よろしいですか？`
        );
        if (!ok) {
          progress.textContent = '処理をキャンセルしました';
          return;
        }

        for (let i = 0; i < records.length; i++) {
          progress.textContent = `再取得中… ${i + 1} / ${records.length}`;

          const result = await refetchLookup(records[i], fieldCode);

          if (!result.success) {
            if (result.error && result.error.code === 'GAIA_LO03') {
              progress.className = 'lookup-error';
              progress.textContent =
                '再取得できませんでした。' +
                'ルックアップ元のコピー元フィールドで' +
                '「値の重複を禁止する」設定が必要です。';
              hasError = true;
              break;
            }
            throw result.error;
          }

          updatedCount++;
        }

        /* ---- 完了メッセージ ---- */
        if (!hasError) {
          progress.className = 'lookup-complete';
          progress.textContent =
            `再取得が完了しました（${updatedCount}件を更新）`;
        }

      } catch (e) {
        console.error(e);
        progress.className = 'lookup-error';
        progress.textContent = '再取得中にエラーが発生しました';
      } finally {
        button.disabled = false;
      }
    };
  });

  /* =========================
   * ルックアップフィールド取得
   * ========================= */
  async function fetchLookupFields() {
    const resp = await kintone.api(
      kintone.api.url('/k/v1/app/form/fields', true),
      'GET',
      { app: kintone.app.getId() }
    );

    return Object.entries(resp.properties)
      .filter(([, f]) => f.lookup)
      .map(([code, f]) => ({
        code,
        label: f.label
      }));
  }

  function populateSelect(select, fields) {
    select.innerHTML = '<option value="">ルックアップを選択</option>';
    fields.forEach(f => {
      const o = document.createElement('option');
      o.value = f.code;
      o.textContent = f.label;
      select.appendChild(o);
    });
  }

  /* =========================
   * キー指定取得
   * ========================= */
  async function fetchRecordsByKey(fieldCode, keyValue) {
    const records = [];
    let offset = 0;
    const limit = 500;
    const query = `${fieldCode} = "${keyValue}"`;

    while (true) {
      const resp = await kintone.api(
        kintone.api.url('/k/v1/records', true),
        'GET',
        {
          app: kintone.app.getId(),
          query: `${query} limit ${limit} offset ${offset}`
        }
      );

      records.push(...resp.records);
      if (resp.records.length < limit) break;
      offset += limit;
    }
    return records;
  }

  /* =========================
   * 再取得処理（エラー判別）
   * ========================= */
  async function refetchLookup(record, lookupFieldCode) {
    const value = record[lookupFieldCode]?.value;
    if (!value) return { success: true };

    try {
      await kintone.api(
        kintone.api.url('/k/v1/record', true),
        'PUT',
        {
          app: kintone.app.getId(),
          id: record.$id.value,
          record: {
            [lookupFieldCode]: { value }
          }
        }
      );
      return { success: true };
    } catch (e) {
      return { success: false, error: e };
    }
  }

})();
