(function () {
  'use strict';

  const PLUGIN_ID = kintone.$PLUGIN_ID;
  const config = kintone.plugin.app.getConfig(PLUGIN_ID);
  const saved = config.settings ? JSON.parse(config.settings) : null;

  let fields = {};

  /* ---------------------------------------------
       1. フィールド情報取得（安定動作方式）
  --------------------------------------------- */
  function fetchFields() {
    return kintone.api(
      kintone.api.url('/k/v1/app/form/fields', true),
      'GET',
      { app: kintone.app.getId() }
    );
  }

  /* ---------------------------------------------
       2. 対象フィールドセレクト描画
  --------------------------------------------- */
  function renderTargetFieldSelect() {
    const select = document.getElementById('target-field');
    select.innerHTML = '';

    Object.keys(fields).forEach(code => {
      const f = fields[code];

      // 対象型に MULTI_LINE_TEXT + RADIO_BUTTON を追加
      if ([
        'SINGLE_LINE_TEXT',
        'MULTI_LINE_TEXT',
        'DROP_DOWN',
        'CHECK_BOX',
        'RADIO_BUTTON'
      ].includes(f.type)) {

        const op = document.createElement('option');
        op.value = code;
        op.textContent = `${f.label}（${code}）`;
        select.appendChild(op);
      }
    });

    if (saved) {
      select.value = saved.targetField;
    }

    // フィールド切り替え時はルールをクリア
    select.addEventListener('change', () => {
      document.getElementById('rule-list').innerHTML = '';
    });
  }

  /* ---------------------------------------------
       3. ルールUI追加
  --------------------------------------------- */
  function addRuleRow(savedRule) {
    const code = document.getElementById('target-field').value;
    if (!code) {
      alert('先に対象フィールドを選択してください');
      return;
    }

    const type = fields[code].type;
    const options = fields[code].options || {};

    const list = document.getElementById('rule-list');

    const row = document.createElement('div');
    row.className = 'rule-row';

    /* ▼ 値入力エリア */
    let valueInput;

    // ✔ 選択肢フィールド（ドロップダウン／チェックボックス／ラジオ）
    if (['DROP_DOWN', 'CHECK_BOX', 'RADIO_BUTTON'].includes(type)) {
      valueInput = document.createElement('select');

      // 空白判定用
      const empty = document.createElement('option');
      empty.value = '';
      empty.textContent = '（空白）';
      valueInput.appendChild(empty);

      // APIの options には label が入っている
      Object.keys(options).forEach(k => {
        const op = document.createElement('option');
        op.value = options[k].label;
        op.textContent = options[k].label;
        valueInput.appendChild(op);
      });

      valueInput.value = savedRule ? savedRule.value : '';

    } else {
      // ✔ 文字列1行／複数行 → テキストボックス
      valueInput = document.createElement('input');
      valueInput.type = 'text';
      valueInput.placeholder = '（部分一致）';
      valueInput.value = savedRule ? savedRule.value : '';
    }

    /* ▼ カラーピッカー */
    const colorPicker = document.createElement('input');
    colorPicker.type = 'color';
    colorPicker.value = savedRule ? savedRule.color : '#ffffff';

    /* ▼ 削除ボタン */
    const delBtn = document.createElement('button');
    delBtn.textContent = '削除';
    delBtn.onclick = () => row.remove();

    /* ▼ 行へ追加 */
    row.appendChild(valueInput);
    row.appendChild(colorPicker);
    row.appendChild(delBtn);

    list.appendChild(row);
  }

  /* ---------------------------------------------
       4. 設定保存
  --------------------------------------------- */
  function saveConfig() {
    const fieldCode = document.getElementById('target-field').value;
    if (!fieldCode) {
      alert('対象フィールドを選択してください');
      return;
    }

    const mode = document.querySelector('input[name="color-mode"]:checked').value;

    const rows = document.querySelectorAll('.rule-row');
    const rules = [];

    rows.forEach(r => {
      const value = r.querySelector('input[type=text], select').value;
      const color = r.querySelector('input[type=color]').value;

      // 空白は value="" でもOK → 色でも判定する
      rules.push({ value, color });
    });

    const settings = {
      targetField: fieldCode,
      fieldType: fields[fieldCode].type,
      colorMode: mode,
      rules
    };

    kintone.plugin.app.setConfig(
      { settings: JSON.stringify(settings) },
      () => {
        alert('設定を保存しました');
        window.location.href = '../../flow?app=' + kintone.app.getId();
      }
    );
  }

  /* ---------------------------------------------
       5. 初期化（検索プラグインと同じ方式）
  --------------------------------------------- */
  fetchFields().then(resp => {
    fields = resp.properties;
    renderTargetFieldSelect();

    if (saved && saved.rules) {
      saved.rules.forEach(rule => addRuleRow(rule));
    }
  });

  /* ---------------------------------------------
       6. ボタンイベント
  --------------------------------------------- */
  document.getElementById('add-rule').onclick = () => addRuleRow();
  document.getElementById('submit').onclick = saveConfig;
  document.getElementById('cancel').onclick = () =>
    window.location.href = '../../flow?app=' + kintone.app.getId();

})();
