(function () {
    'use strict';

    const pluginId = kintone.$PLUGIN_ID;
    const config = kintone.plugin.app.getConfig(pluginId);
    if (!config || !config.fields) return;

    const targetFields = JSON.parse(config.fields);
    if (targetFields.length === 0) return;

    // -----------------------------
    // ヘッダー表示イベント
    // -----------------------------
    kintone.events.on('app.record.index.show', async () => {

        // すでに UI が存在するなら生成しない
        if (document.getElementById('field-quick-search-container')) return;

        const parent = kintone.app.getHeaderSpaceElement();
        if (!parent) return;

        // アプリのフィールド情報取得
        const fieldsInfo = await kintone.api(
            kintone.api.url('/k/v1/app/form/fields', true),
            'GET',
            { app: kintone.app.getId() }
        );

        // コンテナ作成
        const container = document.createElement('div');
        container.id = 'field-quick-search-container';

        // UI情報保存
        const inputs = [];

        // ==============================
        // 🔷 フィールド UI 生成
        // ==============================
        targetFields.forEach((code) => {

            const prop = fieldsInfo.properties[code];
            if (!prop) return;

            const wrapper = document.createElement('div');
            wrapper.className = 'fqs-field-wrapper';

            const label = document.createElement('div');
            label.className = 'fqs-label';
            label.textContent = prop.label;

            let input;

            // -------------------------------------------
            // 🔷 DROP_DOWN → マルチ選択 UI
            // -------------------------------------------
            if (prop.type === 'DROP_DOWN') {

                const box = document.createElement('div');
                box.className = 'fqs-multi-box';

                const inner = document.createElement('div');
                inner.className = 'fqs-multi-inner';

                Object.keys(prop.options).forEach(k => {
                    const v = prop.options[k].label;

                    const item = document.createElement('div');
                    item.className = 'fqs-multi-item';
                    item.textContent = v;

                    item.onclick = () => {
                        item.classList.toggle('selected');
                    };

                    inner.appendChild(item);
                });

                box.appendChild(inner);
                input = box;

            } else {

                // 文字列1行 or 文字列複数行
                input = document.createElement('input');
                input.type = 'text';
                input.className = 'fqs-input';
            }

            wrapper.appendChild(label);
            wrapper.appendChild(input);
            container.appendChild(wrapper);

            inputs.push({ code, type: prop.type, input });
        });


        // ========================================
        // 🔥 stripSelf（超安定版：括弧構造崩さない）
        // ========================================
        const stripSelf = (query) => {
            if (!query) return "";

            let s = query;

            inputs.forEach(({ code }) => {

                const esc = code.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");

                // LIKE 削除（前後の and/or も吸収）
                s = s.replace(
                    new RegExp(
                        `(\\s*(and|or)\\s*)?\\(?\\s*${esc}\\s+like\\s+"[^"]*"\\s*\\)?(\\s*(and|or)\\s*)?`,
                        "gi"
                    ),
                    " "
                );

                // IN 削除
                s = s.replace(
                    new RegExp(
                        `(\\s*(and|or)\\s*)?\\(?\\s*${esc}\\s+in\\s*\\([^)]*\\)\\s*\\)?(\\s*(and|or)\\s*)?`,
                        "gi"
                    ),
                    " "
                );
            });

            // AND/OR 重複を除去
            s = s.replace(/\b(and|or)\b\s*(?=\))/gi, '');
            s = s.replace(/\(\s*(and|or)\s*/gi, '(');
            s = s.replace(/(and|or)\s*(?=\))/gi, '');
            s = s.replace(/\(\s*\)/g, '');

            // 多重スペース除去
            s = s.replace(/\s{2,}/g, ' ').trim();

            // 文頭文末の and/or 除去
            s = s.replace(/^(and|or)\s+/i, '');
            s = s.replace(/\s+(and|or)$/i, '');

            return s.trim();
        };


        // ========================================
        // 🔷 条件生成（あなたの方式で安定）
        // ========================================
        const buildConditions = () => {
            const conds = [];

            inputs.forEach(({ code, type, input }) => {

                if (type === 'DROP_DOWN') {

                    const selected = [...input.querySelectorAll('.fqs-multi-item.selected')]
                        .map(n => n.textContent);

                    if (selected.length) {
                        const c = `${code} in (${selected.map(v => `"${v}"`).join(',')})`;
                        conds.push(`(${c})`);
                    }

                } else {
                    const v = (input.value || '').trim();
                    if (v) {
                        const c = `${code} like "%${v}%"`;
                        conds.push(`(${c})`);
                    }
                }
            });

            return conds.join(' and ');
        };


        // ========================================
        // 🔷 検索ボタン
        // ========================================
        const btnSearch = document.createElement('button');
        btnSearch.textContent = '検索';
        btnSearch.className = 'fqs-btn fqs-btn-search';

        btnSearch.onclick = async () => {

            const base = stripSelf(kintone.app.getQueryCondition() || "");
            const user = buildConditions();

            const finalQ =
                user ? (base ? `(${base}) and (${user})` : user) : base;

            // 余計な and 整形
            const norm = finalQ.replace(/\s+and\s+/gi, ' and ').trim();

            const url = new URL(location.href);
            if (norm) url.searchParams.set('query', norm);
            else url.searchParams.delete('query');

            location.href = url.toString();
        };
// ========================================
// 🔷 Enterキーで検索（★ここに入れる）
// ========================================
inputs.forEach(({ type, input }) => {

  // テキスト入力のみ対象
  if (type === 'DROP_DOWN') return;

  input.addEventListener('keydown', (e) => {
    if (e.key === 'Enter') {
      e.preventDefault();
      btnSearch.onclick();
    }
  });
});

        // ========================================
        // 🔷 クリアボタン
        // ========================================
        const btnClear = document.createElement('button');
        btnClear.textContent = 'クリア';
        btnClear.className = 'fqs-btn fqs-btn-clear';

        btnClear.onclick = () => {

            inputs.forEach(({ input, type }) => {
                if (type === 'DROP_DOWN') {
                    input.querySelectorAll('.fqs-multi-item').forEach(item =>
                        item.classList.remove('selected')
                    );
                } else {
                    input.value = '';
                }
            });

            const base = stripSelf(kintone.app.getQueryCondition() || "");

            const safe = base.trim();
            const url = new URL(location.href);

            if (!safe || /^and$/i.test(safe)) url.searchParams.delete('query');
            else url.searchParams.set('query', safe);

            location.href = url.toString();
        };


        // ========================================
        // 🔷 UIに復元（あなた方式）
        // ========================================
        const queryText = kintone.app.getQueryCondition() || "";

        inputs.forEach(({ code, type, input }) => {

            const esc = code.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");

            // --- DROP_DOWN（in形式）
            if (type === 'DROP_DOWN') {

                const m = queryText.match(new RegExp(`${esc}\\s+in\\s*\\(([^)]*)\\)`, "i"));
                if (m && m[1]) {
                    const vals = m[1].split(',')
                        .map(s => s.replace(/"/g, '').trim());

                    input.querySelectorAll('.fqs-multi-item').forEach(item => {
                        if (vals.includes(item.textContent)) {
                            item.classList.add('selected');
                        }
                    });
                }

            } else {

                // --- LIKE("%xxx%")
                const m = queryText.match(new RegExp(`${esc}\\s+like\\s+"%([^"]*)%"`, "i"));
                if (m) input.value = m[1];
            }
        });


        // 最後に UI を配置
        container.appendChild(btnSearch);
        container.appendChild(btnClear);
        parent.appendChild(container);

    });

})();
