(function () {
  'use strict';

  const MAX_FIELDS = 10;
  const pluginId = kintone.$PLUGIN_ID;
  const config = kintone.plugin.app.getConfig(pluginId);
  const savedFields = config.fields ? JSON.parse(config.fields) : [];

  // --- 日本語ラベル表記 ---
  const TYPE_LABELS = {
    'SINGLE_LINE_TEXT': '文字列1行',
    'MULTI_LINE_TEXT': '文字列複数行',
    'DROP_DOWN': 'ドロップダウン'
  };

  // --- 対象フィールド取得 ---
  function fetchFields() {
    return kintone.api(kintone.api.url('/k/v1/app/form/fields', true), 'GET', {
      app: kintone.app.getId()
    });
  }

  // --- 画面描画 ---
  function renderFieldList(fields) {
    const container = document.getElementById('field-list');
    container.innerHTML = '';

    const items = [];

    Object.values(fields.properties).forEach((prop) => {
      if (
        prop.type === 'SINGLE_LINE_TEXT' ||
        prop.type === 'MULTI_LINE_TEXT' ||
        prop.type === 'DROP_DOWN'
      ) {
        items.push({
          label: prop.label,
          code: prop.code,
          type: prop.type
        });
      }
    });

    // ラベル順に並べる
    items.sort((a, b) => a.label.localeCompare(b.label, 'ja'));

    items.forEach((field) => {
      const wrapper = document.createElement('div');
      wrapper.className = 'field-item';

      const input = document.createElement('input');
      input.type = 'checkbox';
      input.value = field.code;
      input.className = 'cfg-checkbox';

      if (savedFields.includes(field.code)) {
        input.checked = true;
      }

      const label = document.createElement('label');
      label.textContent = `${field.label}（${TYPE_LABELS[field.type]}）`;

      wrapper.appendChild(input);
      wrapper.appendChild(label);
      container.appendChild(wrapper);
    });

    updateDisableStatus();
    setCheckboxEvents();
  }

  // --- 10個選択したら他を disable ---
  function updateDisableStatus() {
    const checkboxes = document.querySelectorAll('.cfg-checkbox');
    const selectedCount = Array.from(checkboxes).filter(cb => cb.checked).length;

    checkboxes.forEach(cb => {
      if (!cb.checked && selectedCount >= MAX_FIELDS) {
        cb.disabled = true;
      } else {
        cb.disabled = false;
      }
    });
  }

  // --- チェックボックスイベント設定 ---
  function setCheckboxEvents() {
    const checkboxes = document.querySelectorAll('.cfg-checkbox');
    checkboxes.forEach(cb => {
      cb.addEventListener('change', updateDisableStatus);
    });
  }

  // --- 保存 ---
  function saveConfig() {
    const checkboxes = document.querySelectorAll('.cfg-checkbox');
    const selected = Array.from(checkboxes).filter(cb => cb.checked).map(cb => cb.value);

    if (selected.length > MAX_FIELDS) {
      alert(`最大 ${MAX_FIELDS} 個まで選択できます。`);
      return;
    }

    kintone.plugin.app.setConfig(
      { fields: JSON.stringify(selected) },
      () => {
        alert('設定を保存しました。');
        window.location.href = '../../flow?app=' + kintone.app.getId();
      }
    );
  }

  // --- キャンセル ---
  function cancelConfig() {
    window.location.href = '../../flow?app=' + kintone.app.getId();
  }

  // --- 初期化 ---
  fetchFields().then(renderFieldList);
  document.getElementById('submit').addEventListener('click', saveConfig);
  document.getElementById('cancel').addEventListener('click', cancelConfig);

})();
