(function () {
    'use strict';

    const pluginId = kintone.$PLUGIN_ID;
    const config = kintone.plugin.app.getConfig(pluginId);
    if (!config || !config.rules) return;

    const rules = JSON.parse(config.rules);

    const EVENTS = [
        'app.record.create.submit',
        'app.record.edit.submit',
        'app.record.index.edit.submit'
    ];

    kintone.events.on(EVENTS, function (event) {
        const record = event.record;

        rules.forEach(rule => {
            if (evaluateRule(rule, record)) {
                applyAction(rule.action, record);
            }
        });

        return event;
    });

    /************************************
     * 1. AND / OR
     ************************************/
    function evaluateRule(rule, record) {
        return rule.logic === 'AND'
            ? rule.conditions.every(cond => evaluateCondition(cond, record))
            : rule.conditions.some(cond => evaluateCondition(cond, record));
    }

    /************************************
     * 2. 条件評価（null / 空 判定対応）
     ************************************/
    function evaluateCondition(cond, record) {
        const field = record[cond.field];
        if (!field) return false;

        const cur = getFieldValue(field);
        const op = cond.operator;
        const val = cond.value;

        /* ===== 空判定専用 ===== */
        if (op === 'IS_EMPTY' || op === 'IS_NOT_EMPTY') {
            const isEmpty =
                cur === null ||
                cur === undefined ||
                cur === '' ||
                (Array.isArray(cur) && cur.length === 0);

            return op === 'IS_EMPTY' ? isEmpty : !isEmpty;
        }

        switch (field.type) {
            case 'SINGLE_LINE_TEXT':
            case 'MULTI_LINE_TEXT':
            case 'LINK':
            case 'DROP_DOWN':
            case 'RADIO_BUTTON':
                return compareString(cur, op, val);

            case 'NUMBER':
            case 'CALC':
                return compareNumber(cur, op, val);

            case 'DATE':
            case 'DATETIME':
                return compareDate(cur, op, val);

            case 'CHECK_BOX':
                return compareCheckboxLabel(cur, op, val);

            default:
                return false;
        }
    }

    /************************************
     * 3. 文字列
     ************************************/
    function compareString(cur, op, val) {
        if (cur === null || cur === undefined) return false;
        if (val === null || val === undefined) return false;

        const left = String(cur);
        const right = String(val);

        switch (op) {
            case '=': return left === right;
            case '≠': return left !== right;
            case 'LIKE': return left.includes(right);
            case 'NOT LIKE': return !left.includes(right);
        }
        return false;
    }

    /************************************
     * 4. 数値（null安全）
     ************************************/
    function compareNumber(cur, op, val) {
        if (cur === null || cur === undefined) return false;
        if (val === null || val === undefined) return false;

        const left = Number(cur);
        const right = Number(val);

        if (Number.isNaN(left) || Number.isNaN(right)) return false;

        switch (op) {
            case '=': return left === right;
            case '≠': return left !== right;
            case '≤': return left <= right;
            case '≥': return left >= right;
        }
        return false;
    }

    /************************************
     * 5. 日付比較（null安全）
     ************************************/
    function compareDate(cur, op, val) {
        if (!cur || !val) return false;

        const left = new Date(cur).getTime();
        const right = new Date(val).getTime();

        if (Number.isNaN(left) || Number.isNaN(right)) return false;

        switch (op) {
            case '=': return left === right;
            case '≠': return left !== right;
            case '<': return left < right;
            case '>': return left > right;
            case '≤': return left <= right;
            case '≥': return left >= right;
        }
        return false;
    }

    /************************************
     * 6. チェックボックス（単一扱い）
     ************************************/
    function compareCheckboxLabel(curArray, op, val) {
        if (!Array.isArray(curArray) || curArray.length === 0) return false;
        if (val === null || val === undefined) return false;

        const cur = curArray[0];

        switch (op) {
            case '=': return cur === val;
            case '≠': return cur !== val;
        }
        return false;
    }

    /************************************
     * 7. 値取り出し
     ************************************/
    function getFieldValue(field) {
        switch (field.type) {
            case 'CHECK_BOX':
                return field.value || [];
            default:
                return field.value ?? null;
        }
    }

    /************************************
     * 8. アクション実行（null = 空にする）
     ************************************/
    function applyAction(action, record) {
        const field = record[action.target];
        if (!field) return;

        const val = action.setValue;

        // null → フィールドを空にする
        if (val === null) {
            field.value =
                field.type === 'CHECK_BOX' ? [] : null;
            return;
        }

        switch (field.type) {
            case 'SINGLE_LINE_TEXT':
            case 'MULTI_LINE_TEXT':
            case 'LINK':
            case 'DROP_DOWN':
            case 'RADIO_BUTTON':
            case 'NUMBER':
            case 'DATE':
            case 'DATETIME':
                field.value = val;
                break;

            case 'CHECK_BOX':
                field.value = [val];
                break;
        }
    }

})();
