(function () {
  'use strict';

  const PLUGIN_ID = kintone.$PLUGIN_ID;
  const config = kintone.plugin.app.getConfig(PLUGIN_ID);
  const savedRules = config.rules ? JSON.parse(config.rules) : [];
  let fields = {};

  /*************************************************
   * 1. フィールド情報取得
   *************************************************/
  function fetchFields() {
    return kintone.api(
      kintone.api.url('/k/v1/app/form/fields', true),
      'GET',
      { app: kintone.app.getId() }
    );
  }

  /*************************************************
   * 2. kintone システム項目の判定
   *************************************************/
  function isSystemField(f) {
    return (
      f.type === 'RECORD_NUMBER' ||
      f.type === 'CREATOR' ||
      f.type === 'CREATED_TIME' ||
      f.type === 'MODIFIER' ||
      f.type === 'UPDATED_TIME'
    );
  }

  /*************************************************
   * 3. 条件側フィールド選択肢
   *************************************************/
  function populateFieldOptions_forCondition(select) {
    select.innerHTML = '';

    Object.keys(fields).forEach(code => {
      const f = fields[code];

      if (
        f.type === 'STATUS' ||
        f.type === 'ASSIGNEE' ||
        f.type === 'STATUS_ASSIGNEE' ||
        f.type === 'USER_SELECT' ||
        f.type === 'CREATOR' ||
        f.type === 'MODIFIER' ||
        f.type === 'CREATED_TIME' ||
        f.type === 'UPDATED_TIME'
      ) return;

      const opt = document.createElement('option');
      opt.value = code;
      opt.textContent = `${f.label} (${code})`;
      select.appendChild(opt);
    });
  }

  /*************************************************
   * 4. アクション側フィールド選択肢
   *************************************************/
  function populateFieldOptions_forAction(select) {
    select.innerHTML = '';

    const editableTypes = [
      'SINGLE_LINE_TEXT',
      'MULTI_LINE_TEXT',
      'NUMBER',
      'DATE',
      'DATETIME',
      'DROP_DOWN',
      'RADIO_BUTTON',
      'CHECK_BOX'
    ];

    Object.keys(fields).forEach(code => {
      const f = fields[code];

      if (!editableTypes.includes(f.type)) return;
      if (isSystemField(f)) return;
      if (f.lookup && f.lookup.fieldMapping) return;
      if (
        f.type === 'USER_SELECT' ||
        f.type === 'GROUP_SELECT' ||
        f.type === 'ORGANIZATION_SELECT'
      ) return;

      const opt = document.createElement('option');
      opt.value = code;
      opt.textContent = `${f.label} (${code})`;
      select.appendChild(opt);
    });
  }

  /*************************************************
   * 5. 条件側 演算子（空判定含む）
   *************************************************/
  function populateOperatorOptions(select, type) {
    select.innerHTML = '';

    let ops = [];

    switch (type) {
      case 'SINGLE_LINE_TEXT':
      case 'MULTI_LINE_TEXT':
      case 'LINK':
      case 'RICH_TEXT':
        ops = [
          ['=', '等しい'],
          ['≠', '等しくない'],
          ['LIKE', '含む'],
          ['NOT LIKE', '含まない'],
          ['IS_EMPTY', '空である'],
          ['IS_NOT_EMPTY', '空でない']
        ];
        break;

      case 'NUMBER':
      case 'CALC':
        ops = [
          ['=', '等しい'],
          ['≠', '等しくない'],
          ['≤', '以下'],
          ['≥', '以上'],
          ['IS_EMPTY', '空である'],
          ['IS_NOT_EMPTY', '空でない']
        ];
        break;

      case 'DATE':
      case 'DATETIME':
        ops = [
          ['=', '等しい'],
          ['≠', '等しくない'],
          ['<', 'より前'],
          ['>', 'より後'],
          ['≤', '以前'],
          ['≥', '以降'],
          ['IS_EMPTY', '空である'],
          ['IS_NOT_EMPTY', '空でない']
        ];
        break;

      case 'DROP_DOWN':
      case 'RADIO_BUTTON':
      case 'CHECK_BOX':
        ops = [
          ['=', '等しい'],
          ['≠', '等しくない'],
          ['IS_EMPTY', '空である'],
          ['IS_NOT_EMPTY', '空でない']
        ];
        break;

      default:
        ops = [['=', '等しい']];
        break;
    }

    ops.forEach(([v, label]) => {
      const o = document.createElement('option');
      o.value = v;
      o.textContent = label;
      select.appendChild(o);
    });
  }

  /*************************************************
   * 6. 条件値 UI
   *************************************************/
  function renderConditionValue(wrapper, field, savedValue) {
    wrapper.innerHTML = '';

    const type = field.type;
    const options = field.options || {};

    if (type === 'DROP_DOWN' || type === 'RADIO_BUTTON' || type === 'CHECK_BOX') {
      const s = document.createElement('select');
      Object.keys(options).forEach(k => {
        const op = document.createElement('option');
        op.value = options[k].label;
        op.textContent = options[k].label;
        if (savedValue === options[k].label) op.selected = true;
        s.appendChild(op);
      });
      wrapper.appendChild(s);
      return;
    }

    if (type === 'DATE' || type === 'DATETIME') {
      const inp = document.createElement('input');
      inp.type = 'date';
      if (savedValue) inp.value = savedValue;
      wrapper.appendChild(inp);
      return;
    }

    const inp = document.createElement('input');
    inp.type = (type === 'NUMBER' || type === 'CALC') ? 'number' : 'text';
    if (savedValue !== null && savedValue !== undefined) inp.value = savedValue;
    wrapper.appendChild(inp);
  }

  /*************************************************
   * 7. アクション値 UI
   *************************************************/
  function renderActionValue(wrapper, field, savedValue) {
    wrapper.innerHTML = '';
    if (!field) return;

    const type = field.type;
    const options = field.options || {};

    if (type === 'DROP_DOWN' || type === 'RADIO_BUTTON' || type === 'CHECK_BOX') {
      const s = document.createElement('select');
      Object.keys(options).forEach(k => {
        const op = document.createElement('option');
        op.value = options[k].label;
        op.textContent = options[k].label;
        if (savedValue === options[k].label) op.selected = true;
        s.appendChild(op);
      });
      wrapper.appendChild(s);
      return;
    }

    if (type === 'DATE' || type === 'DATETIME') {
      const inp = document.createElement('input');
      inp.type = 'date';
      if (savedValue) inp.value = savedValue;
      wrapper.appendChild(inp);
      return;
    }

    const inp = document.createElement('input');
    inp.type = (type === 'NUMBER' || type === 'CALC') ? 'number' : 'text';
    if (savedValue !== null && savedValue !== undefined) inp.value = savedValue;
    wrapper.appendChild(inp);
  }

  /*************************************************
   * 8. 条件行追加
   *************************************************/
  function addConditionRow(container, conditionTemplate, savedCond) {
    const frag = conditionTemplate.content.cloneNode(true);
    const row = frag.querySelector('.condition-row');

    const fieldSelect = row.querySelector('.cond-field');
    const opSelect = row.querySelector('.cond-operator');
    const valueArea = row.querySelector('.cond-value');

    populateFieldOptions_forCondition(fieldSelect);

    fieldSelect.addEventListener('change', () => {
      const f = fields[fieldSelect.value];
      populateOperatorOptions(opSelect, f.type);
      renderConditionValue(valueArea, f, null);
    });

    opSelect.addEventListener('change', () => {
      const isNullOp =
        opSelect.value === 'IS_EMPTY' ||
        opSelect.value === 'IS_NOT_EMPTY';

      const input = valueArea.querySelector('input,select');
      if (input) {
        input.disabled = isNullOp;
        if (isNullOp) input.value = '';
      }
    });

    row.querySelector('.delete-condition-btn')
      .addEventListener('click', () => row.remove());

    if (savedCond) {
      fieldSelect.value = savedCond.field;
      const f = fields[savedCond.field];
      populateOperatorOptions(opSelect, f.type);
      opSelect.value = savedCond.operator;
      renderConditionValue(valueArea, f, savedCond.value);

      const isNullOp =
        savedCond.operator === 'IS_EMPTY' ||
        savedCond.operator === 'IS_NOT_EMPTY';

      const input = valueArea.querySelector('input,select');
      if (input) input.disabled = isNullOp;
    } else {
      const f = fields[fieldSelect.value];
      populateOperatorOptions(opSelect, f.type);
      renderConditionValue(valueArea, f, null);
    }

    container.appendChild(frag);
  }

  /*************************************************
   * 9. ルールカード追加
   *************************************************/
  function addRuleCard(savedRule) {
    const ruleList = document.getElementById('rule-list');
    const ruleTemplate = document.getElementById('rule-template');
    const conditionTemplate = document.getElementById('condition-template');

    const id = 'rule-' + Math.random().toString(36).slice(2);
    const fragment = ruleTemplate.content.cloneNode(true);
    const card = fragment.querySelector('.rule-card');

    const radios = fragment.querySelectorAll('input[type="radio"]');
    radios.forEach(r => { r.name = `logic-${id}`; });

    radios[0].checked = !savedRule || savedRule.logic !== 'OR';
    radios[1].checked = savedRule && savedRule.logic === 'OR';

    const condWrapper = fragment.querySelector('.conditions');

    fragment.querySelector('.add-condition-btn')
      .addEventListener('click', () =>
        addConditionRow(condWrapper, conditionTemplate, null)
      );

    fragment.querySelector('.delete-rule-btn')
      .addEventListener('click', () => card.remove());

    const actionTarget = fragment.querySelector('.action-target');
    const actionValueArea = fragment.querySelector('.action-value');

    populateFieldOptions_forAction(actionTarget);

    actionTarget.addEventListener('change', () => {
      const f = fields[actionTarget.value];
      renderActionValue(actionValueArea, f, null);
    });

    if (savedRule) {
      savedRule.conditions.forEach(c =>
        addConditionRow(condWrapper, conditionTemplate, c)
      );

      actionTarget.value = savedRule.action.target;
      const f = fields[actionTarget.value];
      renderActionValue(actionValueArea, f, savedRule.action.setValue);
    } else {
      addConditionRow(condWrapper, conditionTemplate, null);
    }

    ruleList.appendChild(fragment);
  }

  /*************************************************
   * 10. 保存処理
   *************************************************/
  function saveConfig() {
    const ruleList = document.getElementById('rule-list');
    const cards = ruleList.querySelectorAll('.rule-card');
    const rules = [];

    cards.forEach(card => {
      const logic = card.querySelector('input[type="radio"]:checked').value;

      const conditions = [...card.querySelectorAll('.condition-row')].map(row => {
        const selects = row.querySelectorAll('select');
        const fieldSelect = selects[0];
        const opSelect = selects[1];
        const valueInput = row.querySelector('.cond-value').querySelector('input,select');

        let value = null;

        if (
          opSelect.value !== 'IS_EMPTY' &&
          opSelect.value !== 'IS_NOT_EMPTY' &&
          valueInput
        ) {
          const rawValue = valueInput.value;
          if (rawValue !== '') {
            const field = fields[fieldSelect.value];
            value =
              field && (field.type === 'NUMBER' || field.type === 'CALC')
                ? Number(rawValue)
                : rawValue;
          }
        }

        return {
          field: fieldSelect.value,
          operator: opSelect.value,
          value
        };
      });

      const actionTarget = card.querySelector('.action-target');
      const actionValueInput = card.querySelector('.action-value').querySelector('input,select');

      let actionValue = null;

      if (actionValueInput) {
        const rawValue = actionValueInput.value;
        if (rawValue !== '') {
          const field = fields[actionTarget.value];
          actionValue =
            field && (field.type === 'NUMBER' || field.type === 'CALC')
              ? Number(rawValue)
              : rawValue;
        }
      }

      rules.push({
        logic,
        conditions,
        action: {
          target: actionTarget.value,
          setValue: actionValue
        }
      });
    });

    kintone.plugin.app.setConfig(
      { rules: JSON.stringify(rules) },
      () => {
        alert('設定を保存しました');
        window.location.href = '../../flow?app=' + kintone.app.getId();
      }
    );
  }

  /*************************************************
   * 11. 初期ロード
   *************************************************/
  fetchFields().then(resp => {
    fields = resp.properties;
    if (savedRules.length > 0) {
      savedRules.forEach(r => addRuleCard(r));
    } else {
      addRuleCard(null);
    }
  });

  document.getElementById('add-rule-btn').onclick = () => addRuleCard(null);
  document.getElementById('save-btn').onclick = saveConfig;
  document.getElementById('cancel-btn').onclick = () => {
    window.location.href = '../../flow?app=' + kintone.app.getId();
  };

})();
